package org.msh.tb.adminunits.latlng;

import org.jboss.seam.Component;
import org.jboss.seam.annotations.In;
import org.jboss.seam.annotations.Name;
import org.jboss.seam.annotations.Transactional;
import org.jboss.seam.annotations.async.Asynchronous;
import org.msh.etbm.commons.transactionlog.DetailXMLWriter;
import org.msh.tb.application.TransactionManager;
import org.msh.tb.entities.*;
import org.msh.tb.entities.enums.RoleAction;
import org.msh.utils.JsonUtils;

import javax.persistence.EntityManager;
import java.util.*;

/**
 * Created by mauri on 27/06/2017.
 * Requests google services for lat and lng info and set it to administrative units registries
 */
@Name("lagLngGenerator")
public class LagLngGenerator {

    @In
    EntityManager entityManager;

    protected TransactionManager transaction;

    @Asynchronous
    public void generate() {
        int totalSuccess = 0;
        int totalFailed = 0;
        String auFailed = "";

        // select all administrativeunits from production workspace, levels 1, 2 and 3
        List<AdministrativeUnit> auList = entityManager.createQuery("from AdministrativeUnit au " +
                "where au.workspace.id = 11 " +
                "and au.countryStructure.level in (1,2,3) " +
                "order by au.code").getResultList();

        for (AdministrativeUnit au : auList) {
            if (GoogleApiLatLonService.setLatLgn(au)){
                totalSuccess++;

                // save in database
                if (!getTransaction().isActive()) {
                    getTransaction().begin();
                }

                entityManager.persist(au);
                entityManager.flush();

                getTransaction().commit();
            } else {
                totalFailed++;

                // store for log
                auFailed = auFailed + au.getFullDisplayName2() + " / ";
            }
        }

        // start preparing transaction log
        // SUPPORT_SERVICE
        UserRole userRole = entityManager.find(UserRole.class, 308);
        // System user - as it is running async
        UserLog userLog = entityManager.find(UserLog.class, 11);
        // production workspace
        WorkspaceLog workspaceLog = entityManager.find(WorkspaceLog.class, 11);

        if (userLog == null || userRole == null || workspaceLog == null) {
            return;
        }

        TransactionLog log = new TransactionLog();
        log.setRole(userRole);
        log.setUser(userLog);
        log.setAction(RoleAction.EXEC);
        log.setTransactionDate(new Date());
        log.setWorkspace(workspaceLog);
        log.setTitleSuffix("Load latitude and Longitude from google API");

        DetailXMLWriter detailWriter = new DetailXMLWriter();
        detailWriter.addTable();
        detailWriter.addTableRow("Total", auList.size());
        detailWriter.addTableRow("Success", totalSuccess);
        detailWriter.addTableRow("Failed", totalFailed);
        if (!auFailed.equals("")) {
            detailWriter.addTableRow("Administrative units failed", auFailed.substring(0, auFailed.length()-3));
        }
        log.setComments(detailWriter.asXML());

        // save transaction log
        if (!getTransaction().isActive()) {
            getTransaction().begin();
        }

        entityManager.persist(log);
        entityManager.flush();

        getTransaction().commit();
    }

    /**
     * Return the transaction in use by the task
     * @return
     */
    protected TransactionManager getTransaction() {
        if (transaction == null)
            transaction = (TransactionManager) Component.getInstance("transactionManager");
        return transaction;
    }
}
